<?php
/**
 * Admin settings page: Settings > 2fox4 Accessibility.
 *
 * @package Fox4A11y
 */

namespace Fox4A11y;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers the settings page and fields.
 */
class Settings {

	/** Option group / option name. */
	private const OPTION = 'fox4a11y_settings';

	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Add the submenu page under Settings.
	 */
	public function add_menu(): void {
		add_options_page(
			__( '2fox4 Accessibility', '2fox4-accessibility' ),
			__( '2fox4 Accessibility', '2fox4-accessibility' ),
			'manage_options',
			'fox4a11y',
			array( $this, 'render_page' )
		);
	}

	/**
	 * Register setting, section, and fields.
	 */
	public function register_settings(): void {
		register_setting( 'fox4a11y_group', self::OPTION, array(
			'type'              => 'array',
			'sanitize_callback' => array( $this, 'sanitize' ),
			'default'           => array(
				'accent_color' => '#007AFF',
				'position'     => 'bottom-right',
			),
		) );

		add_settings_section(
			'fox4a11y_main',
			__( 'Widget Appearance', '2fox4-accessibility' ),
			'__return_null',
			'fox4a11y'
		);

		add_settings_field(
			'accent_color',
			__( 'Accent Color', '2fox4-accessibility' ),
			array( $this, 'field_color' ),
			'fox4a11y',
			'fox4a11y_main'
		);

		add_settings_field(
			'position',
			__( 'Widget Position', '2fox4-accessibility' ),
			array( $this, 'field_position' ),
			'fox4a11y',
			'fox4a11y_main'
		);
	}

	/**
	 * Sanitize submitted settings.
	 */
	public function sanitize( $input ): array {
		$clean = array();

		$clean['accent_color'] = sanitize_hex_color( $input['accent_color'] ?? '#007AFF' ) ?: '#007AFF';

		$allowed_positions     = array( 'bottom-right', 'bottom-left' );
		$clean['position']     = in_array( $input['position'] ?? '', $allowed_positions, true )
			? $input['position']
			: 'bottom-right';

		return $clean;
	}

	/**
	 * Color picker field.
	 */
	public function field_color(): void {
		$val = Plugin::get_option( 'accent_color', '#007AFF' );
		printf(
			'<input type="color" name="%s[accent_color]" value="%s" />',
			esc_attr( self::OPTION ),
			esc_attr( $val )
		);
		echo '<p class="description">' . esc_html__( 'Choose the accent color for the accessibility toggle button.', '2fox4-accessibility' ) . '</p>';
	}

	/**
	 * Position select field.
	 */
	public function field_position(): void {
		$val = Plugin::get_option( 'position', 'bottom-right' );
		printf(
			'<select name="%s[position]">',
			esc_attr( self::OPTION )
		);
		$positions = array(
			'bottom-right' => __( 'Bottom Right', '2fox4-accessibility' ),
			'bottom-left'  => __( 'Bottom Left', '2fox4-accessibility' ),
		);
		foreach ( $positions as $key => $label ) {
			printf(
				'<option value="%s" %s>%s</option>',
				esc_attr( $key ),
				selected( $val, $key, false ),
				esc_html( $label )
			);
		}
		echo '</select>';
	}

	/**
	 * Render the settings page.
	 */
	public function render_page(): void {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( '2fox4 Accessibility Suite', '2fox4-accessibility' ); ?></h1>
			<form method="post" action="options.php">
				<?php
				settings_fields( 'fox4a11y_group' );
				do_settings_sections( 'fox4a11y' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}
}
