<?php
/**
 * Frontend: enqueue assets and render the floating widget markup.
 *
 * @package Fox4A11y
 */

namespace Fox4A11y;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Outputs the accessibility widget on the frontend.
 */
class Frontend {

	public function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue' ) );
		add_action( 'wp_footer', array( $this, 'render_widget' ) );
	}

	/**
	 * Enqueue CSS and JS.
	 */
	public function enqueue(): void {
		wp_enqueue_style(
			'fox4a11y-widget',
			FOX4A11Y_URL . 'assets/css/widget.css',
			array(),
			FOX4A11Y_VERSION
		);

		wp_enqueue_script(
			'fox4a11y-widget',
			FOX4A11Y_URL . 'assets/js/widget.js',
			array(),
			FOX4A11Y_VERSION,
			true
		);

		$accent   = Plugin::get_option( 'accent_color', '#007AFF' );
		$position = Plugin::get_option( 'position', 'bottom-right' );

		wp_localize_script( 'fox4a11y-widget', 'fox4a11yConfig', array(
			'accentColor' => sanitize_hex_color( $accent ),
			'position'    => in_array( $position, array( 'bottom-right', 'bottom-left' ), true ) ? $position : 'bottom-right',
			'i18n'        => array(
				'en' => array(
					'title'            => __( 'Accessibility Settings', '2fox4-accessibility' ),
					'fontSize'         => __( 'Font Size', '2fox4-accessibility' ),
					'dyslexiaFont'     => __( 'Dyslexia Font', '2fox4-accessibility' ),
					'highContrast'     => __( 'High Contrast', '2fox4-accessibility' ),
					'darkMode'         => __( 'Dark Mode', '2fox4-accessibility' ),
					'readingGuide'     => __( 'Reading Guide', '2fox4-accessibility' ),
					'highlightLinks'   => __( 'Highlight Links', '2fox4-accessibility' ),
					'keyboardNav'      => __( 'Keyboard Navigation', '2fox4-accessibility' ),
					'reset'            => __( 'Reset All', '2fox4-accessibility' ),
					'close'            => __( 'Close', '2fox4-accessibility' ),
					'openAccessibility'=> __( 'Open Accessibility Settings', '2fox4-accessibility' ),
				),
				'de' => array(
					'title'            => 'Barrierefreiheit',
					'fontSize'         => 'Schriftgröße',
					'dyslexiaFont'     => 'Legasthenie-Schrift',
					'highContrast'     => 'Hoher Kontrast',
					'darkMode'         => 'Dunkelmodus',
					'readingGuide'     => 'Lesehilfe',
					'highlightLinks'   => 'Links hervorheben',
					'keyboardNav'      => 'Tastaturnavigation',
					'reset'            => 'Zurücksetzen',
					'close'            => 'Schließen',
					'openAccessibility'=> 'Barrierefreiheit öffnen',
				),
			),
		) );
	}

	/**
	 * Render the widget shell in the footer. JS populates content.
	 */
	public function render_widget(): void {
		$position = Plugin::get_option( 'position', 'bottom-right' );
		$accent   = Plugin::get_option( 'accent_color', '#007AFF' );
		?>
		<div id="fox4a11y-root"
			 class="fox4a11y-pos-<?php echo esc_attr( $position ); ?>"
			 data-accent="<?php echo esc_attr( $accent ); ?>"
			 aria-label="<?php esc_attr_e( 'Accessibility Tools', '2fox4-accessibility' ); ?>">

			<!-- Toggle button -->
			<button id="fox4a11y-toggle"
					type="button"
					aria-expanded="false"
					aria-controls="fox4a11y-panel"
					aria-label="<?php esc_attr_e( 'Open Accessibility Settings', '2fox4-accessibility' ); ?>"
					style="background-color:<?php echo esc_attr( $accent ); ?>">
				<svg aria-hidden="true" focusable="false" width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
					<circle cx="12" cy="4.5" r="2.5"/>
					<path d="M12 7v5"/>
					<path d="M8 11l4 2 4-2"/>
					<path d="M9 20l3-8 3 8"/>
				</svg>
			</button>

			<!-- Panel -->
			<div id="fox4a11y-panel"
				 role="dialog"
				 aria-modal="true"
				 aria-label="<?php esc_attr_e( 'Accessibility Settings', '2fox4-accessibility' ); ?>"
				 hidden>
				<div class="fox4a11y-panel-inner">

					<header class="fox4a11y-header">
						<h2 id="fox4a11y-title"><?php esc_html_e( 'Accessibility Settings', '2fox4-accessibility' ); ?></h2>
						<button id="fox4a11y-close"
								type="button"
								aria-label="<?php esc_attr_e( 'Close', '2fox4-accessibility' ); ?>">
							<svg aria-hidden="true" focusable="false" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
								<line x1="18" y1="6" x2="6" y2="18"/>
								<line x1="6" y1="6" x2="18" y2="18"/>
							</svg>
						</button>
					</header>

					<div class="fox4a11y-body">

						<!-- Font size -->
						<div class="fox4a11y-control fox4a11y-control--range">
							<label for="fox4a11y-fontsize" data-i18n="fontSize"><?php esc_html_e( 'Font Size', '2fox4-accessibility' ); ?></label>
							<div class="fox4a11y-range-wrap">
								<button type="button" class="fox4a11y-step" data-dir="-1" aria-label="<?php esc_attr_e( 'Decrease font size', '2fox4-accessibility' ); ?>">A&minus;</button>
								<input id="fox4a11y-fontsize" type="range" min="80" max="200" step="10" value="100" aria-valuenow="100" aria-valuemin="80" aria-valuemax="200" />
								<button type="button" class="fox4a11y-step" data-dir="1" aria-label="<?php esc_attr_e( 'Increase font size', '2fox4-accessibility' ); ?>">A+</button>
								<output for="fox4a11y-fontsize" class="fox4a11y-output">100%</output>
							</div>
						</div>

						<!-- Toggles -->
						<div class="fox4a11y-control fox4a11y-control--toggle">
							<span data-i18n="dyslexiaFont"><?php esc_html_e( 'Dyslexia Font', '2fox4-accessibility' ); ?></span>
							<button type="button" role="switch" aria-checked="false" data-feature="dyslexia" aria-label="<?php esc_attr_e( 'Dyslexia Font', '2fox4-accessibility' ); ?>">
								<span class="fox4a11y-switch-knob"></span>
							</button>
						</div>

						<div class="fox4a11y-control fox4a11y-control--toggle">
							<span data-i18n="highContrast"><?php esc_html_e( 'High Contrast', '2fox4-accessibility' ); ?></span>
							<button type="button" role="switch" aria-checked="false" data-feature="high-contrast" aria-label="<?php esc_attr_e( 'High Contrast', '2fox4-accessibility' ); ?>">
								<span class="fox4a11y-switch-knob"></span>
							</button>
						</div>

						<div class="fox4a11y-control fox4a11y-control--toggle">
							<span data-i18n="darkMode"><?php esc_html_e( 'Dark Mode', '2fox4-accessibility' ); ?></span>
							<button type="button" role="switch" aria-checked="false" data-feature="dark-mode" aria-label="<?php esc_attr_e( 'Dark Mode', '2fox4-accessibility' ); ?>">
								<span class="fox4a11y-switch-knob"></span>
							</button>
						</div>

						<div class="fox4a11y-control fox4a11y-control--toggle">
							<span data-i18n="readingGuide"><?php esc_html_e( 'Reading Guide', '2fox4-accessibility' ); ?></span>
							<button type="button" role="switch" aria-checked="false" data-feature="reading-guide" aria-label="<?php esc_attr_e( 'Reading Guide', '2fox4-accessibility' ); ?>">
								<span class="fox4a11y-switch-knob"></span>
							</button>
						</div>

						<div class="fox4a11y-control fox4a11y-control--toggle">
							<span data-i18n="highlightLinks"><?php esc_html_e( 'Highlight Links', '2fox4-accessibility' ); ?></span>
							<button type="button" role="switch" aria-checked="false" data-feature="highlight-links" aria-label="<?php esc_attr_e( 'Highlight Links', '2fox4-accessibility' ); ?>">
								<span class="fox4a11y-switch-knob"></span>
							</button>
						</div>

						<div class="fox4a11y-control fox4a11y-control--toggle">
							<span data-i18n="keyboardNav"><?php esc_html_e( 'Keyboard Navigation', '2fox4-accessibility' ); ?></span>
							<button type="button" role="switch" aria-checked="false" data-feature="keyboard-nav" aria-label="<?php esc_attr_e( 'Keyboard Navigation', '2fox4-accessibility' ); ?>">
								<span class="fox4a11y-switch-knob"></span>
							</button>
						</div>

					</div>

					<footer class="fox4a11y-footer">
						<button id="fox4a11y-reset" type="button" data-i18n="reset">
							<?php esc_html_e( 'Reset All', '2fox4-accessibility' ); ?>
						</button>
						<p class="fox4a11y-credit">
							Designed by <a href="https://www.2fox4.de/wordpress-plugins/" target="_blank" rel="noopener noreferrer">2fox4</a>
						</p>
					</footer>

				</div>
			</div>

		</div>
		<?php
	}
}
