/**
 * 2fox4 Accessibility Suite — Frontend Widget
 *
 * Vanilla JS. No jQuery. Detects browser language and applies
 * localised labels. Manages features via <html> class names and
 * persists state in localStorage.
 *
 * @package Fox4A11y
 */
( function () {
	'use strict';

	/* ==================================================================
	   Constants & State
	   ================================================================== */
	var STORAGE_KEY = 'fox4a11y_state';
	var config      = window.fox4a11yConfig || {};
	var i18nStrings = config.i18n || {};

	/** Detect language — prefer 2-char code, fall back to 'en'. */
	function detectLang() {
		var raw  = ( navigator.language || navigator.userLanguage || 'en' ).toLowerCase();
		var code = raw.split( '-' )[0];
		return i18nStrings[ code ] ? code : 'en';
	}

	var lang = detectLang();
	var t    = i18nStrings[ lang ] || i18nStrings.en || {};

	/* ==================================================================
	   Helpers
	   ================================================================== */
	function $( selector, ctx ) {
		return ( ctx || document ).querySelector( selector );
	}

	function $$( selector, ctx ) {
		return Array.prototype.slice.call( ( ctx || document ).querySelectorAll( selector ) );
	}

	function loadState() {
		try {
			return JSON.parse( localStorage.getItem( STORAGE_KEY ) ) || {};
		} catch ( e ) {
			return {};
		}
	}

	function saveState( state ) {
		try {
			localStorage.setItem( STORAGE_KEY, JSON.stringify( state ) );
		} catch ( e ) {
			/* storage full — fail silently */
		}
	}

	/* ==================================================================
	   Initialisation
	   ================================================================== */
	function init() {
		var root   = $( '#fox4a11y-root' );
		if ( ! root ) return;

		/*
		 * Re-parent the widget directly onto <html> (documentElement) — NOT <body>.
		 *
		 * WHY: Several of our own features apply CSS filter / transform to <body>
		 * (e.g. high-contrast: filter:contrast(1.5)). Any filter/transform/
		 * perspective/will-change on an ancestor creates a new Containing Block,
		 * which makes position:fixed relative to THAT ancestor instead of the
		 * viewport. By placing the widget on <html> (which never gets those
		 * properties) we guarantee it stays viewport-fixed at all times.
		 */
		if ( root.parentNode !== document.documentElement ) {
			document.documentElement.appendChild( root );
		}

		var toggle = $( '#fox4a11y-toggle' );
		var panel  = $( '#fox4a11y-panel' );
		var close  = $( '#fox4a11y-close' );
		var reset  = $( '#fox4a11y-reset' );
		var slider = $( '#fox4a11y-fontsize' );
		var output = $( '.fox4a11y-output', root );
		var steps  = $$( '.fox4a11y-step', root );

		/* ---------- Apply accent color as CSS variable ---------- */
		if ( config.accentColor ) {
			document.documentElement.style.setProperty( '--fox4a11y-accent', config.accentColor );
			toggle.style.backgroundColor = config.accentColor;
		}

		/* ---------- Localise labels ---------- */
		applyTranslations( root );

		/* ---------- Restore persisted state ---------- */
		var state = loadState();
		restoreFeatures( state, slider, output );

		/* ---------- Panel open/close ---------- */
		toggle.addEventListener( 'click', function () {
			var expanded = panel.hasAttribute( 'hidden' );
			if ( expanded ) {
				openPanel( toggle, panel );
			} else {
				closePanel( toggle, panel );
			}
		} );

		close.addEventListener( 'click', function () {
			closePanel( toggle, panel );
		} );

		/* Close on Escape */
		document.addEventListener( 'keydown', function ( e ) {
			if ( e.key === 'Escape' && ! panel.hasAttribute( 'hidden' ) ) {
				closePanel( toggle, panel );
			}
		} );

		/* Close when clicking outside */
		document.addEventListener( 'mousedown', function ( e ) {
			if ( ! panel.hasAttribute( 'hidden' ) && ! root.contains( e.target ) ) {
				closePanel( toggle, panel );
			}
		} );

		/* ---------- Font-size slider ---------- */
		slider.addEventListener( 'input', function () {
			applyFontSize( slider.value, output );
		} );

		/* Step buttons (A- / A+) */
		steps.forEach( function ( btn ) {
			btn.addEventListener( 'click', function () {
				var dir = parseInt( btn.getAttribute( 'data-dir' ), 10 );
				var val = parseInt( slider.value, 10 ) + dir * parseInt( slider.step, 10 );
				val = Math.max( parseInt( slider.min, 10 ), Math.min( parseInt( slider.max, 10 ), val ) );
				slider.value = val;
				slider.dispatchEvent( new Event( 'input' ) );
			} );
		} );

		/* ---------- Toggle switches ---------- */
		$$( '[role="switch"]', root ).forEach( function ( sw ) {
			sw.addEventListener( 'click', function () {
				var checked = sw.getAttribute( 'aria-checked' ) !== 'true';
				sw.setAttribute( 'aria-checked', String( checked ) );
				applyFeature( sw.getAttribute( 'data-feature' ), checked );
			} );
		} );

		/* ---------- Reset ---------- */
		reset.addEventListener( 'click', function () {
			resetAll( root, slider, output );
		} );

		/* ---------- Reading guide mousemove ---------- */
		document.addEventListener( 'mousemove', handleReadingGuide );
	}

	/* ==================================================================
	   Panel helpers
	   ================================================================== */
	function openPanel( toggle, panel ) {
		panel.removeAttribute( 'hidden' );
		toggle.setAttribute( 'aria-expanded', 'true' );
		/* Trap focus — move to first focusable */
		var first = panel.querySelector( 'button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])' );
		if ( first ) first.focus();
	}

	function closePanel( toggle, panel ) {
		panel.setAttribute( 'hidden', '' );
		toggle.setAttribute( 'aria-expanded', 'false' );
		toggle.focus();
	}

	/* ==================================================================
	   Translations
	   ================================================================== */
	function applyTranslations( root ) {
		$$( '[data-i18n]', root ).forEach( function ( el ) {
			var key   = el.getAttribute( 'data-i18n' );
			var label = t[ key ];
			if ( label ) el.textContent = label;
		} );

		/* Panel title */
		var title = $( '#fox4a11y-title', root );
		if ( title && t.title ) title.textContent = t.title;

		/* Toggle aria-label */
		var toggle = $( '#fox4a11y-toggle', root );
		if ( toggle && t.openAccessibility ) toggle.setAttribute( 'aria-label', t.openAccessibility );

		/* Close aria-label */
		var closebtn = $( '#fox4a11y-close', root );
		if ( closebtn && t.close ) closebtn.setAttribute( 'aria-label', t.close );

		/* Switch aria-labels */
		$$( '[role="switch"]', root ).forEach( function ( sw ) {
			var key   = sw.getAttribute( 'data-feature' );
			var map   = {
				'dyslexia':        'dyslexiaFont',
				'high-contrast':   'highContrast',
				'dark-mode':       'darkMode',
				'reading-guide':   'readingGuide',
				'highlight-links': 'highlightLinks',
				'keyboard-nav':    'keyboardNav'
			};
			var label = t[ map[ key ] ];
			if ( label ) sw.setAttribute( 'aria-label', label );
		} );
	}

	/* ==================================================================
	   Feature Application
	   ================================================================== */
	var featureClassMap = {
		'dyslexia':        'fox4a11y-dyslexia',
		'high-contrast':   'fox4a11y-high-contrast',
		'dark-mode':       'fox4a11y-dark-mode',
		'reading-guide':   'fox4a11y-reading-guide',
		'highlight-links': 'fox4a11y-highlight-links',
		'keyboard-nav':    'fox4a11y-keyboard-nav'
	};

	function applyFeature( feature, on ) {
		var cls = featureClassMap[ feature ];
		if ( ! cls ) return;

		document.documentElement.classList.toggle( cls, on );

		/* Reading guide element management */
		if ( feature === 'reading-guide' ) {
			toggleReadingGuide( on );
		}

		persistState();
	}

	function applyFontSize( value, output ) {
		/*
		 * Apply font-size scaling to <body> instead of <html>.
		 * Setting it on <html> would affect ALL rem-based layout in the
		 * entire theme and can shift our fixed-position widget.
		 * Our widget uses px sizing and is immune to this change.
		 */
		document.body.style.fontSize = value + '%';
		if ( output ) output.textContent = value + '%';
		var slider = $( '#fox4a11y-fontsize' );
		if ( slider ) slider.setAttribute( 'aria-valuenow', value );
		persistState();
	}

	/* ==================================================================
	   Reading Guide
	   ================================================================== */
	var guideEl = null;

	function toggleReadingGuide( on ) {
		if ( on ) {
			if ( ! guideEl ) {
				guideEl = document.createElement( 'div' );
				guideEl.id = 'fox4a11y-reading-guide';
				guideEl.setAttribute( 'aria-hidden', 'true' );
				document.body.appendChild( guideEl );
			}
			guideEl.style.display = 'block';
		} else if ( guideEl ) {
			guideEl.style.display = 'none';
		}
	}

	function handleReadingGuide( e ) {
		if ( guideEl && guideEl.style.display !== 'none' ) {
			guideEl.style.top = ( e.clientY - 6 ) + 'px';
		}
	}

	/* ==================================================================
	   Persistence
	   ================================================================== */
	function persistState() {
		var state = {};
		var slider = $( '#fox4a11y-fontsize' );
		if ( slider ) state.fontSize = slider.value;

		$$( '[role="switch"]' ).forEach( function ( sw ) {
			state[ sw.getAttribute( 'data-feature' ) ] = sw.getAttribute( 'aria-checked' ) === 'true';
		} );

		saveState( state );
	}

	function restoreFeatures( state, slider, output ) {
		if ( ! state || Object.keys( state ).length === 0 ) return;

		/* Font size */
		if ( state.fontSize ) {
			slider.value = state.fontSize;
			applyFontSize( state.fontSize, output );
		}

		/* Toggle switches */
		Object.keys( featureClassMap ).forEach( function ( feature ) {
			if ( state[ feature ] ) {
				var sw = $( '[data-feature="' + feature + '"]' );
				if ( sw ) {
					sw.setAttribute( 'aria-checked', 'true' );
					applyFeature( feature, true );
				}
			}
		} );
	}

	/* ==================================================================
	   Reset
	   ================================================================== */
	function resetAll( root, slider, output ) {
		/* Font size */
		slider.value = 100;
		applyFontSize( 100, output );

		/* Switches */
		$$( '[role="switch"]', root ).forEach( function ( sw ) {
			sw.setAttribute( 'aria-checked', 'false' );
			applyFeature( sw.getAttribute( 'data-feature' ), false );
		} );

		/* Clear classes */
		Object.values( featureClassMap ).forEach( function ( cls ) {
			document.documentElement.classList.remove( cls );
		} );

		document.body.style.fontSize = '';

		/* Clear storage */
		try {
			localStorage.removeItem( STORAGE_KEY );
		} catch ( e ) {
			/* silent */
		}
	}

	/* ==================================================================
	   Boot
	   ================================================================== */
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', init );
	} else {
		init();
	}
} )();
